/**
 * \file: levelchanger.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: authorization level daemon
 *
 * \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
 *
 * \copyright (c) 2010, 2011 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/
#include <sys/eventfd.h>
#include <unistd.h>
#include <pthread.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <libgen.h>
#include <sys/wait.h>
#include <dirent.h>
#include <sys/stat.h>
#include <errno.h>

#include "control/levelchanger.h"
#include "control/daemon_fsm.h"
#include "control/configuration.h"
#include "control/ald.h"
#include "model/signature_db.h"
#include "model/daemon_model.h"
#include "util/logger.h"

//normally this: eg: WEXITSTATUS(exit_code_raw),lin(t) has problems with this macro
#define ALD_WTERMSIG(x) ((x) & 0x7f)
#define ALD_WEXITSTATUS(x) (((x) >> 8) & 0xff)
#define ALD_WIFEXITED(x) (ALD_WTERMSIG (x) == 0)

static bool levelchanger_verify_all_files(char *path);
static bool levelchanger_does_script_exist_in_script_folder(char *path);
static bool levelchanger_verify_feature_files(const char *feature_name);

//---------------------------------------- private structures -------------------------------------------------------
typedef enum thread_ctx_state_t
{
	IDLE=0,
	PREPARED,
	WORKING
} thread_ctx_state_t;

typedef struct thread_ctx_t
{
	bool daemon_exiting;

	thread_ctx_state_t state;

	error_code_t levelchange_result;

	pthread_t thread_id;

	level_configuration_t *level_configuration;

	change_type_t change_type;

	pid_t execute_scripts_prc_id;
} thread_ctx_t;

typedef enum ald_script_kind_t
{
	PREPARE = 0,
	FINALIZE
}ald_script_kind_t;
//-------------------------------------------------------------------------------------------------------------------

//---------------------------------------- private attributes -------------------------------------------------------
static int levelchanger_eventfd=-1;
static thread_ctx_t thread_ctx={0};
static const char *root_dir;
static pthread_mutex_t execution_preparation_mutex = PTHREAD_MUTEX_INITIALIZER;
//-------------------------------------------------------------------------------------------------------------------

//---------------------------------------- private functions --------------------------------------------------------
static void levelchanger_signal_job_done_to_mainloop(void);
static void levelchanger_reset_job_done_signal(void);
static void *levelchanger_thread_func(void *data);

static bool levelchanger_verify_level_conf(security_level_t level_number);

static bool levelchanger_verify_script_integrity(const char *path);

static void levelchanger_execute_script(char *path);
static void levelchanger_execute_levelscripts(void);

static void levelchanger_execute_prescript(void);
static void levelchanger_execute_permanent_script(const feature_t *feature,const char *feature_name,
		bool is_active,bool is_permanent_level);
static void levelchanger_execute_volatile_script(const feature_t *feature,const char *feature_name,	bool is_active);
static void levelchanger_execute_postscript(void);

//-------------------------------------------------------------------------------------------------------------------

//---------------------------------------- API members --------------------------------------------------------------
error_code_t levelchanger_init(void)
{
	levelchanger_eventfd=eventfd(0,EFD_CLOEXEC|EFD_NONBLOCK);
	if (levelchanger_eventfd==-1)
		return RESULT_NORESOURCE;
	thread_ctx.state=IDLE;
	thread_ctx.thread_id=0;
	thread_ctx.daemon_exiting=false;
	thread_ctx.execute_scripts_prc_id = 0;
	root_dir=NULL;
	return RESULT_OK;
}

void levelchanger_deinit(void)
{
	pthread_mutex_lock(&execution_preparation_mutex);
	thread_ctx.daemon_exiting=true;
	pthread_mutex_unlock(&execution_preparation_mutex);

	if (levelchanger_eventfd==-1)
		close(levelchanger_eventfd);

	levelchanger_eventfd=-1;
	root_dir=NULL;
}

error_code_t levelchanger_prepare_levelchange(security_level_t targeted_level, bool *is_permanent_ptr)
{
	error_code_t result;

	if (thread_ctx.state!=IDLE)
	{
		logger_log_error("LEVELCHANGER - Calling prepare in state different from IDLE is not allowed. Implementation error!!");
		return RESULT_INVALID;
	}

	root_dir=configuration_get_script_root_dir();
	logger_log_debug("LEVELCHANGER - Verifying the level configuration file of level %d.",(int)targeted_level);
	if (!levelchanger_verify_level_conf(targeted_level))
	{
		logger_log_error("Verification of the configuration file of level %d failed.", (int)targeted_level);
		return RESULT_LEVEL_COMFIGURATION_MODIFIED;
	}

	logger_log_debug("LEVELCHANGER - Loading the level configuration file of level %d.",(int)targeted_level);
	result=level_configuration_load_level(&thread_ctx.level_configuration,root_dir,targeted_level);
	if (result==RESULT_OK)
	{
		if (is_permanent_ptr!=NULL)
			*is_permanent_ptr=level_configuration_is_permanent_level(thread_ctx.level_configuration);
	}

	thread_ctx.state=PREPARED;

	return result;
}

void levelchanger_unprepare_levelchange(void)
{
	if (thread_ctx.state==PREPARED)
	{
		if (thread_ctx.level_configuration!=NULL)
			level_configuration_free(thread_ctx.level_configuration);
		thread_ctx.state=IDLE;
	}
}

error_code_t levelchanger_kickoff_levelchange(change_type_t type)
{
	if (thread_ctx.state==WORKING)
	{
		logger_log_error("LEVELCHANGER - Trying to kick off the worker thread twice which"
				" is not allowed. Implementation error!!");
		return RESULT_INVALID;
	}
	else if (thread_ctx.state==IDLE)
	{
		logger_log_error("LEVELCHANGER - Trying to kick off the worker thread without"
				" preparing it. Implementation error!!");
		return RESULT_INVALID;
	}

	thread_ctx.levelchange_result=RESULT_OK;
	thread_ctx.change_type=type;

	thread_ctx.state=WORKING;

	logger_log_debug("LEVELCHANGER - Start changing to level %d.",
			level_configuration_get_security_level(thread_ctx.level_configuration));

	//kick off a thread
	if (pthread_create(&thread_ctx.thread_id,NULL, levelchanger_thread_func, NULL)!=0)
		return RESULT_NORESOURCE;

	//from here on, don't touch thread_ctx anymore because the thread is now working with the data structure
	return RESULT_OK;
}

int levelchanger_get_pollfd(void)
{
	return levelchanger_eventfd;
}

void levelchanger_on_event(void)
{
	//acknowledging the signal
	levelchanger_reset_job_done_signal();

	if (thread_ctx.state != WORKING)
	{
		logger_log_error("LEVELCHANGER - Level changer got an event from the mainloop"
				" without having a thread working. Ignoring it.");
		return;
	}

	//here, the worker thread signaled us that it is not working any longer. So we can work on thread_ctx again.
	logger_log_debug("LEVELCHANGER - Level changer got an event from the mainloop."
			" Sending the 'level change complete' signal to the FSM.");

	//marking the levelchanger as non working
	thread_ctx.state=IDLE;

	//cleaning up
	level_configuration_free(thread_ctx.level_configuration);
	thread_ctx.level_configuration=NULL;

	daemon_fsm_signal_level_change_complete(thread_ctx.levelchange_result);
}
//-------------------------------------------------------------------------------------------------

//------------------------------------- private members -------------------------------------------
static void levelchanger_signal_job_done_to_mainloop(void)
{
	uint64_t event_incr=1;
	logger_log_debug("LEVELCHANGER - Signaling the mainloop that we are done.");
	if (write(levelchanger_eventfd,&event_incr,sizeof(uint64_t))==0)
	{

	}
}

static void levelchanger_reset_job_done_signal(void)
{
	uint64_t event_incr=1;
	logger_log_debug("LEVELCHANGER - Reseting our signal that brought us here.");
	if (read(levelchanger_eventfd,&event_incr,sizeof(uint64_t))==0)
	{

	}
}

static void *levelchanger_thread_func(void *data)
{
	//don't need this parameter
	(void)data;

	//- at this point, the daemon fsm ensures that the main thread keeps of its hands from thread_ctx so we can use it here
	//- the logger is mutexed so we can easily log from here and from the main thread at the same time
	//- configuration data (root_dir) and the signature db is critical during shutdown only.
	//	The access to them is mutexed by execution_preparation_mutex and thread_ctx.daemon_exiting
	//		-> in case thread_ctx.daemon_exiting is true, nothing is touched any more
	//		-> the evaluation of the flag and the access to the data is done by the thread within a locked mutex
	//		-> the flag is switched by the daemon with a locked mutex as well

	//executing pre script
	levelchanger_execute_prescript();

	//executing the feature scripts
	logger_log_debug("LEVELCHANGER - Executing the feature scripts.");
	levelchanger_execute_levelscripts();

	//executing post script
	levelchanger_execute_postscript();

	pthread_mutex_lock(&execution_preparation_mutex);
	if (!thread_ctx.daemon_exiting)
		levelchanger_signal_job_done_to_mainloop();
	//from here on, don't touch thread_ctx anymore since we are here signaling the main thread that it can go
	//on working on and evaluating the attributes.
	pthread_mutex_unlock(&execution_preparation_mutex);

	pthread_exit(NULL);

	return NULL;
}

pid_t levelchanger_get_script_exec_prc_id()
{
	return thread_ctx.execute_scripts_prc_id;
}
void levelchanger_set_result_to_script_exec_timeout(void)
{
	thread_ctx.levelchange_result=RESULT_SCRIPT_EXEC_TIMEOUT;
}

static void levelchanger_execute_script(char *path)
{
	pid_t childprc = fork();

	if (childprc > 0)
	{
		ald_activate_script_exec_timeout();

		logger_log_debug("LEVELCHANGER - Created child process for script execution");
		/*This is the Parent*/
		int status=0;
		int exit_code=0;

		thread_ctx.execute_scripts_prc_id = childprc;

	    /* PRQA: Lint Message 64: Type mismatch (arg. no. 1) (union __WAIT_STATUS = int *) */
		wait(&status);/*lint !e64 */

		/* PRQA: Lint Message 774: The value of status changes in wait*/
		if (ALD_WIFEXITED(status))/*lint !e774 */
		{
			/* PRQA: Lint Message 774: The value of status changes in wait*/
			exit_code=ALD_WEXITSTATUS(status); /*lint !e774 */

			/* PRQA: Lint Message 774: The value of status changes in wait  */
			if (exit_code!=0) /*lint !e774 */
			{
				logger_log_error("The script \'%s\' returned with a non zero exit code. Exit code: %d",path,exit_code);
					thread_ctx.levelchange_result=RESULT_SCRIPTS_FAILED;
			}
			logger_log_debug("LEVELCHANGER - Child exited with exit_code \'%d\'.", exit_code);
		}
		else
		{
			logger_log_debug("LEVELCHANGER - The child process for script execution was killed by the signal: %d",ALD_WTERMSIG(status));
		}
		ald_deactivate_script_exec_timeout();
		thread_ctx.execute_scripts_prc_id = 0;
	}
	else if (childprc == 0)
	{
		char* argv[1];
		argv[0] = NULL;
		char path_cpy[PATH_MAX];
		char *dir;
		int exit_code_raw = 0;

		strncpy(path_cpy, path,PATH_MAX);
		dir=dirname(path_cpy);
		setenv("ALD_CUR_SCRIPT_EXEC_PATH",dir,1);

		exit_code_raw = execv(path,argv);
		exit(exit_code_raw);
	}
	else
	{
		logger_log_error("Child process (for script execution) creation failed");
		logger_log_errmem("ALD - Child process (for script execution) creation failed");
		thread_ctx.levelchange_result=RESULT_SCRIPTS_FAILED;
	}
}

static bool levelchanger_verify_script_integrity(const char *path)
{
	if (!signature_db_is_script_available(path))
	{
		logger_log_errmem("LEVELCHANGER - Script \'%s\' is available, "
				"but it's entry not found in sigdb.", path);
		return false;
	}

	if (!signature_db_validate_script(path))
	{
		logger_log_errmem("LEVELCHANGER - Script \'%s\' modified.", path);
		thread_ctx.levelchange_result=RESULT_SCRIPTS_MODIFIED;
		return false;
	}

	return true;
}

static void levelchanger_execute_pre_or_post_script(ald_script_kind_t script_kind)
{
	char path[PATH_MAX]={0};
	bool can_execute = false;
	struct stat stat_result;

	//executing the prepare script
	pthread_mutex_lock(&execution_preparation_mutex);
	if (!thread_ctx.daemon_exiting)
	{
		switch(script_kind)
		{
			case PREPARE:
				logger_log_debug("LEVELCHANGER - Executing the prepare script.");
				level_configuration_get_prepare_script_path(path, PATH_MAX, thread_ctx.level_configuration,
						thread_ctx.change_type, root_dir);
				break;
			case FINALIZE:
				logger_log_debug("LEVELCHANGER - Executing the finalize script.");
				level_configuration_get_finalize_script_path(path, PATH_MAX, thread_ctx.level_configuration,
						thread_ctx.change_type, root_dir);
				break;
		}

		if (lstat(path,&stat_result) == -1)
		{
			can_execute = false;
			if (signature_db_is_script_available(path))
				logger_log_errmem("WARNING: File \'%s\' not found,but it's entry found in sigdb",path);
		}
		else if (S_ISREG(stat_result.st_mode))
			can_execute = levelchanger_verify_script_integrity(path);
		else
		{
			can_execute = false;
			logger_log_errmem("WARNING: Unexpected file type for file \'%s\' - Expected regular file type",path);
		}
	}
	else
		can_execute = false;
	pthread_mutex_unlock(&execution_preparation_mutex);

	if (can_execute)
		levelchanger_execute_script(path);
}

static void levelchanger_execute_prescript(void)
{
	levelchanger_execute_pre_or_post_script(PREPARE);
}

static void levelchanger_execute_permanent_script(const feature_t *feature,const char *feature_name,
		bool is_active,bool is_permanent_level)
{
	char path[PATH_MAX]={0};
	struct stat stat_result;

	level_configuration_get_permanent_feature_script_path(path,PATH_MAX,feature, root_dir, is_permanent_level);

	// Ignoring the case, if a script is present during verification but missing during execution.
	if (stat(path,&stat_result) == 0)
	{
		//execute permanent scripts (used for permanent levels during normal change and recovery)
		if (S_ISDIR(stat_result.st_mode))
		{
			if (is_active && is_permanent_level)
			{
				strncat(path,FEATURE_ENABLE_PERM_FN,PATH_MAX-1);
			}
			else
			{
				strncat(path,FEATURE_DISABLE_PERM_FN,PATH_MAX-1);
			}
		}
		logger_log_debug("LEVELCHANGER - Executing permanent %s script: %s.",feature_name,path);
		levelchanger_execute_script(path);
	}
}

static bool levelchanger_does_script_exist_in_script_folder(char *path)
{
	char script_path[PATH_MAX];
	struct stat stat_result;
	bool can_execute=false;

	snprintf(script_path,PATH_MAX,"%s/%s.sh",path,basename(path));

	if (stat(script_path,&stat_result)==-1)
		logger_log_errmem("WARNING: \'%s\' script not found in the script folder- %s.", script_path,strerror(errno));
	else
		can_execute=true;

	return can_execute;
}

static bool levelchanger_verify_all_files(char *path)
{
	bool can_execute=true;
	struct dirent *ep;
	DIR *dp;
	char verify_file[PATH_MAX];

	can_execute=levelchanger_does_script_exist_in_script_folder(path);
	if(can_execute==true)
	{
		dp = opendir (path);
		if (dp == NULL)
		{
			logger_log_info("WARNING: Unable to open the Script dir in \'%s\'- %s.\n",path,strerror(errno));
			can_execute=false;
			return can_execute;
		}

		while ((ep = readdir (dp))!=NULL && can_execute==true)
		{
			if (strcmp(ep->d_name,"..")==0 || strcmp(ep->d_name,".")==0) continue;

			if (ep->d_type != DT_REG)
			{
				logger_log_errmem("WARNING: Unexpected file type for file \'%s/%s\' - Expected regular file type",path,ep->d_name);
				can_execute = false;
				break;
			}
			logger_log_debug("File \'%s\' found inside \'%s\'.",ep->d_name,path);
			snprintf(verify_file,PATH_MAX,"%s/%s",path,ep->d_name);
			can_execute=levelchanger_verify_script_integrity(verify_file);
		}
		closedir(dp);
	}
	return can_execute;
}

static void levelchanger_execute_volatile_script(const feature_t *feature,const char *feature_name,	bool is_active)
{
	char path[PATH_MAX]={0};
	struct stat stat_result;

	level_configuration_get_volatile_feature_script_path(path,PATH_MAX,feature, root_dir);

	// Ignoring the case, if a script is present during verification but missing during execution.
	if (stat(path,&stat_result) == 0)
	{
		if (S_ISDIR(stat_result.st_mode))
		{
			if (is_active)
			{
				strncat(path,FEATURE_ENABLE_VOLA_FN,PATH_MAX-1);
			}
			else
			{
				strncat(path,FEATURE_DISABLE_VOLA_FN,PATH_MAX-1);
			}
		}
		logger_log_debug("LEVELCHANGER - Executing volatile %s script: %s.",feature_name,path);
		levelchanger_execute_script(path);
	}
}

static void levelchanger_execute_postscript(void)
{
	levelchanger_execute_pre_or_post_script(FINALIZE);
}

static void levelchanger_execute_levelscripts()
{
	const feature_t *feature;
	change_type_t type;
	bool is_permanent_level;
	bool can_execute=false;

	type=thread_ctx.change_type;
	is_permanent_level=level_configuration_is_permanent_level(thread_ctx.level_configuration);

	feature=level_configuration_get_first_feature(thread_ctx.level_configuration);
	while (feature != NULL && !thread_ctx.daemon_exiting && (thread_ctx.levelchange_result!=RESULT_SCRIPT_EXEC_TIMEOUT))
	{
		bool is_active;
		const char *feature_name;

		is_active=level_configuration_is_feature_active(feature);
		feature_name=level_configuration_feature_name(feature);

		logger_log_debug("LEVELCHANGER - Processing scripts for %s.",feature_name);
		can_execute = levelchanger_verify_feature_files(feature_name);

		if(can_execute)
		{
			//executing the permanent scripts
			levelchanger_execute_permanent_script(feature,feature_name,is_active,is_permanent_level);

			if(thread_ctx.levelchange_result!=RESULT_SCRIPT_EXEC_TIMEOUT)
			{
				//execute volatile scripts (executed in volatile and permanent change but not during recovery)
				if (type!=ALD_RECOVERY_CHANGE)
					levelchanger_execute_volatile_script(feature,feature_name,is_active);
			}
		}
		else
			logger_log_errmem("LEVELCHANGER - Skipping feature execution for %s.",feature_name);

		feature=level_configuration_next_feature(feature);
	}
}

static bool levelchanger_verify_level_conf(security_level_t level_number)
{
	char level_conf_path[PATH_MAX];
	snprintf(level_conf_path, PATH_MAX, "%s%s/%d/%s",root_dir, LEVEL_SUBDIR,(int)level_number,LEVEL_CONF_FN);

	return signature_db_validate_script(level_conf_path);
}

static bool levelchanger_verify_feature_files(const char *feature_name)
{
	int idx = 0;
	bool can_execute = false;
	struct stat stat_result;
	char path[PATH_MAX]={0};
	char dir_script[PATH_MAX]={0};
	const char *feature_toggle_value[] = {
			FEATURE_ENABLE_PERM_DIR,
			FEATURE_DISABLE_PERM_DIR,
			FEATURE_ENABLE_VOLA_DIR,
			FEATURE_DISABLE_VOLA_DIR
	};
	int feature_toggle_count = ARRAY_SIZE(feature_toggle_value);

	pthread_mutex_lock(&execution_preparation_mutex);
	if (!thread_ctx.daemon_exiting)
	{
		for(idx = 0; idx < feature_toggle_count ; idx++)
		{
			can_execute = true;
			snprintf(path,PATH_MAX,"%s%s/%s/%s",root_dir,FEATURE_SUBDIR,feature_name,feature_toggle_value[idx]);
/*
 * Common cases:
 * ------------
 * 		Case 1: If a file has bad signature, This is an error/attack. The feature execution should stop.
 * 		Case 2: If a regular file is available, but its entry is not there in signature db. This looks like an
 * 					error/attack. Therefore the feature should not be executed.
 *					Inside feature directory, this case is applicable only for known scripts.
 *					Inside feature sub-directory,this case is applicable to all the files
 *		Case 3: If a file is a symbolic link, then it is an error/attack. Because only regular files are expected.
 *					Therefore the feature should not be executed.
 *
 * Non-directory cases - Inside feature directory but not inside feature sub-directory:
 * -----------------------------------------------------------------------------------
 * 		Case 4: If a single script file is missing and it's entry is not available in the signature db, that can
 * 					 be by intention. Therefore the feature should be executed
 * 		Case 5: If a single script file is missing but it's entry is available in the signature db, this looks like
 * 					 an error/attack. Therefore the feature should not be executed
 * 		Case 6: If an unexpected element whose name is having the same naming scheme as sub-directory is available,
 * 					this looks like an error/attack. Therefore, the feature should not be executed.
 *					e.g. In kernel_console feature, if a file "../security_features/kernel_console/enable_permanent"
 *					 	 exists which is neither a script(i.e. enable_permanent.sh) nor a directory
 *					 	 (i.e. enable_permanent/), then it is an unexpected element
 *
 * Directory cases - Only inside feature sub-directory:
 * ---------------------------------------------------
 *		Case 7: If a script(having same name as directory) is missing inside a directory, this looks like an
 *					error/attack(independent of it's entry in signature db).Therefore the feature should not be executed
 * 		Case 8: If a directory is missing but if the script inside it has an entry in signature db, this looks like an
 * 					error/attack. Therefore the feature should not be executed
 *TODO:	Case 9: If a directory has a files with unknown names (e.g. common.sh) and such a file is missing but it has
 * 					an entry in sigdb. This looks like an error/attack. Therefore the feature should not be executed.
 *
 * 	The different case numbers are commented in the below if conditions in the places where they are done.
 *
 */
			if (lstat(path,&stat_result)==-1)
			{
				if (errno == ENOENT)
				{
					snprintf(dir_script,PATH_MAX,"%s/%s.sh",path,basename(path));
					strncat(path,SCRIPT_EXTN,PATH_MAX-1);

					if (lstat(path,&stat_result)==-1)
					{
						if (signature_db_is_script_available(path)) // case 5
						{
							logger_log_errmem("WARNING: \'%s\' script not found,but it's entry found in sigdb",path);
							can_execute = false;
						}
						else if (signature_db_is_script_available(dir_script)) // case 8
						{
							logger_log_errmem("WARNING: script \'%s\' entry found in sigdb,but the script and its holding"
									" directory not found",dir_script);
							can_execute = false;
						}
						else // case 4
							logger_log_debug("%s doesn't have %s script - Skipping it",feature_name,feature_toggle_value[idx]);
					}
					else if (S_ISREG(stat_result.st_mode)) //case 1 & 2
							can_execute = levelchanger_verify_script_integrity(path);
					else // case 3
					{
						logger_log_errmem("WARNING: Unexpected file type for file \'%s\' - "
								"Expected regular file type",path);
						can_execute = false;
					}
				}
				else
				{
					logger_log_errmem("WARNING: Unexpected stat error for path \'%s\' - %s.",path,strerror(errno));
					can_execute = false;
				}
			}
			else if (S_ISDIR(stat_result.st_mode)) // case 1, 2, 3 & 7
				can_execute = levelchanger_verify_all_files(path);
			else // case 6 & 3
			{
				logger_log_errmem("WARNING: Unexpected element found \'%s\'",path);
				can_execute = false;
			}

			memset(path,0,PATH_MAX);memset(dir_script,0,PATH_MAX);
			if(can_execute == false)
				break;
		}
	}
	pthread_mutex_unlock(&execution_preparation_mutex);

	return can_execute;
}
//-------------------------------------------------------------------------------------------------
